/**
 * 
 */
package gov.va.med.mhv.usermgmt.util;

import gov.va.med.mhv.core.util.Precondition;
import gov.va.med.mhv.core.util.ServiceResponseUtils;
import gov.va.med.mhv.usermgmt.enumeration.ExtractType;
import gov.va.med.mhv.usermgmt.service.AccessControlCollectionServiceResponse;
import gov.va.med.mhv.usermgmt.service.delegate.ServiceDelegateFactory;
import gov.va.med.mhv.usermgmt.transfer.AccessControl;
import gov.va.med.mhv.usermgmt.transfer.Patient;

import java.util.Collection;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.tigris.atlas.messages.MessageUtils;
import org.tigris.atlas.service.BooleanServiceResponse;

/**
 * @author Rob Proper (Aquilent Inc.)
 *
 */
public final class PHRAccessControl {
    
    private static final Log LOG = LogFactory.getLog(PHRAccessControl.class);
    
    private PHRAccessControl() {
    }

    public static BooleanServiceResponse hasPhrAccess(Patient patient) {
        Precondition.assertNotNull("patient", patient);
        Precondition.assertNotNull("patient.userProfile", patient.
            getUserProfile());
        return hasPhrAccess(patient.getUserProfile().getUserName());
    }

    @SuppressWarnings("unchecked")
    public static BooleanServiceResponse hasPhrAccess(String userName) {
        Precondition.assertNotBlank("userName", userName);
        return hasPhrAccess(ExtractTypeUtils.getAllExtractTypes(), userName);
    }

    @SuppressWarnings("unchecked")
    public static BooleanServiceResponse hasPhrAccess(ExtractType extractType, 
        Patient patient) 
    {
        Precondition.assertNotNull("extractType", extractType);
        Precondition.assertNotNull("patient", patient);
        Precondition.assertNotNull("patient.userProfile", patient.
            getUserProfile());
        ExtractType[] extractTypes = { extractType };
        return hasPhrAccess(extractTypes, patient.getUserProfile().
            getUserName());
    }
    
    @SuppressWarnings("unchecked")
    private static BooleanServiceResponse hasPhrAccess(
        ExtractType[] extractTypes, String userName) 
    {
        assert extractTypes != null;
        assert !StringUtils.isBlank(userName);
        BooleanServiceResponse response = new BooleanServiceResponse();

        AccessControlCollectionServiceResponse accessResponse =
            ServiceDelegateFactory.createAccessControlServiceDelegate().
            findAccessControls(userName);
        if (accessResponse.getMessages().hasErrorMessages()) {
            if (LOG.isErrorEnabled()) {
                LOG.error("Unable to determine patient " + userName + 
                    " access to extracts '" + extractTypes + "'");
            }
            ServiceResponseUtils.logMessages(response, PHRAccessControl.class, 
                LOG);
            response.getMessages().addMessage(MessageUtils.createErrorMessage(
                MessageKeys.UNKNOWN_EXCEPTION_OCCURRED, null, null));
            return response;
        } 
        
        response.setBoolean(false);
        Collection<AccessControl> accessControls = accessResponse.
            getAccessControls();
        if (accessControls == null) {
            return response;
        }
        for (ExtractType extractType: extractTypes) {
            if (hasPhrAccess(extractType, userName, accessControls)) { 
                response.setBoolean(true);
                return response;
            }
        }
        return response;
    }
    
    @SuppressWarnings("unchecked")
    private static boolean hasPhrAccess(ExtractType extractType, 
        String userName, Collection<AccessControl> accessControls) 
    {
        assert extractType != null;
        assert !StringUtils.isBlank(userName);
        boolean hasAccess = false;
        if ((accessControls != null)) {
            hasAccess = AccessControlUtils.hasAccess(AccessDomainUtils.PHR, 
                extractType.getName(), AccessPermissionUtils.READ, 
                accessControls);
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("Patient " + userName + " has " +
                (hasAccess ? "" : "NO ") + " access to extract '" +
                extractType.getName() + "'");
        }
        return hasAccess;
    }
    
}
